<?php
/**
 * Fungsi-fungsi utama untuk bot Telegram
 */

/**
 * Koneksi ke database
 */
function connectDB()
{
    try {
    
        if (!isset($_ENV['DB_HOST']) || !isset($_ENV['DB_NAME']) || !isset($_ENV['DB_USER'])) {
            throw new PDOException("Konfigurasi database tidak lengkap");
        }
    
        $db = new PDO(
            "mysql:host=" . $_ENV['DB_HOST'] . ";dbname=" . $_ENV['DB_NAME'] . ";charset=utf8mb4",
            $_ENV['DB_USER'],
            $_ENV['DB_PASS'],
            [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
            ]
        );
        return $db;
    } catch (PDOException $e) {
        logBotActivity('db_connection_error', $e->getMessage());
        die("Database connection failed: " . $e->getMessage());
    }
}

/**
 * Kirim pesan ke Telegram
 */
function sendMessage($chatId, $text, $keyboard = null)
{
    $url = "https://api.telegram.org/bot" . $_ENV['BOT_TOKEN'] . "/sendMessage";
    $params = [
        'chat_id' => $chatId,
        'text' => $text,
        'parse_mode' => 'HTML',
    ];
    
    if ($keyboard !== null) {
        $params['reply_markup'] = json_encode($keyboard);
    }
    
    return makeRequest($url, $params);
}

/**
 * Kirim dokumen ke Telegram
 */
function sendDocument($chatId, $filePath, $caption = null)
{
    $url = "https://api.telegram.org/bot" . $_ENV['BOT_TOKEN'] . "/sendDocument";
    
    $document = new CURLFile($filePath);
    $params = [
        'chat_id' => $chatId,
        'document' => $document,
    ];
    
    if ($caption !== null) {
        $params['caption'] = $caption;
        $params['parse_mode'] = 'HTML';
    }
    
    return makeRequest($url, $params, true);
}

/**
 * Jawab callback query
 */
function answerCallbackQuery($callbackQueryId, $text = null, $showAlert = false)
{
    $url = "https://api.telegram.org/bot" . $_ENV['BOT_TOKEN'] . "/answerCallbackQuery";
    $params = [
        'callback_query_id' => $callbackQueryId,
        'show_alert' => $showAlert,
    ];
    
    if ($text !== null) {
        $params['text'] = $text;
    }
    
    return makeRequest($url, $params);
}

/**
 * Set webhook
 */
function setWebhook($webhookUrl)
{
    $url = "https://api.telegram.org/bot" . $_ENV['BOT_TOKEN'] . "/setWebhook";
    $params = [
        'url' => $webhookUrl,
    ];
    
    return makeRequest($url, $params);
}

/**
 * Melakukan request ke API Telegram
 */
function makeRequest($url, $params, $isMultipart = false)
{
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    
    if ($isMultipart) {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
    } else {
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
    }
    
    $response = curl_exec($ch);
    
    if (curl_errno($ch)) {
        logBotActivity('curl_error', curl_error($ch));
        return false;
    }
    
    curl_close($ch);
    return json_decode($response, true);
}

/**
 * Registrasi user baru
 */
function registerUser($telegramId, $username, $firstName, $lastName, $db)
{
    try {
        // Check if user exists
        $stmt = $db->prepare("SELECT id FROM users WHERE telegram_id = ?");
        $stmt->execute([$telegramId]);
        
        if (!$stmt->fetch()) {
            // Insert new user
            $stmt = $db->prepare("INSERT INTO users (telegram_id, username, first_name, last_name) VALUES (?, ?, ?, ?)");
            $stmt->execute([$telegramId, $username, $firstName, $lastName]);
            logBotActivity('new_user_registered', $telegramId);
        } else {
            // Update user info
            $stmt = $db->prepare("UPDATE users SET username = ?, first_name = ?, last_name = ? WHERE telegram_id = ?");
            $stmt->execute([$username, $firstName, $lastName, $telegramId]);
        }
        
        return true;
    } catch (PDOException $e) {
        logBotActivity('user_registration_error', $e->getMessage());
        return false;
    }
}

/**
 * Cek apakah user diblokir
 */
function isUserBlocked($telegramId, $db)
{
    try {
        $stmt = $db->prepare("SELECT is_blocked FROM users WHERE telegram_id = ?");
        $stmt->execute([$telegramId]);
        $user = $stmt->fetch();
        
        return $user && $user['is_blocked'] == 1;
    } catch (PDOException $e) {
        logBotActivity('check_blocked_error', $e->getMessage());
        return false;
    }
}

/**
 * Cek apakah user memiliki akses ke member area
 */
function hasAccessToMemberArea($telegramId, $db)
{
    try {
        $stmt = $db->prepare("
            SELECT ak.id 
            FROM access_keys ak
            JOIN users u ON ak.user_id = u.id
            WHERE u.telegram_id = ? AND ak.is_used = 1
        ");
        $stmt->execute([$telegramId]);
        
        return $stmt->fetch() ? true : false;
    } catch (PDOException $e) {
        logBotActivity('check_access_error', $e->getMessage());
        return false;
    }
}

/**
 * Proses aktivasi key
 */
function processKeyActivation($key, $chatId, $telegramId, $db)
{
    try {
        $formattedKey = chunk_split($key, 4, '-');
        // Check if key exists and not used
        $stmt = $db->prepare("SELECT id, is_used FROM access_keys WHERE key_value = ?");
        $stmt->execute([$key]);
        $keyData = $stmt->fetch();
        
        if (!$keyData) {
            $message = "❌ <b>UNRECOGNIZED ACCESS KEY</b> ❌\n";
            $message .= str_repeat("═", 30) . "\n";
            $message .= "The key you entered is not registered in our system\n\n";
            $message .= "🔍 <b>Details:</b>\n";
            $message .= "• Key: <code>$formattedKey</code>\n";
            $message .= "• Status: Not found\n\n";
            $message .= "🔄 <b>What to do:</b>\n";
            $message .= "1. Double-check for typos\n";
            $message .= "2. Contact @rinda_escobar for assistance\n";
            $message .= "3. Request a new key if needed\n\n";
            $message .= "💎 <i>Premium membership includes exclusive content and features</i>";
            
            $keyboard = [
                'inline_keyboard' => [
                    [
                        ['text' => '🆘 CONTACT SUPPORT', 'url' => 'https://t.me/rinda_escobar']
                    ]
                ]
            ];
            
            sendMessage($chatId, $message, $keyboard);
            return false;
        }
        
        if ($keyData['is_used'] && $keyData['user_id'] != $userId) {
            $stmt = $db->prepare("SELECT first_name FROM users WHERE id = ?");
            $stmt->execute([$keyData['user_id']]);
            $user = $stmt->fetch();
            
            $message = "🚫 <b>KEY ALREADY REDEEMED</b> 🚫\n";
            $message .= str_repeat("═", 30) . "\n";
            $message .= "This access key was already activated by another user\n\n";
            $message .= "📅 <b>Activation Date:</b> " . date('d M Y', strtotime($keyData['used_at'])) . "\n";
            $message .= "👤 <b>Registered To:</b> " . ($user['first_name'] ?? 'User #'.$keyData['user_id']) . "\n\n";
            $message .= "❗ <b>If this wasn't you:</b>\n";
            $message .= "1. Contact @rinda_escobar immediately\n";
            $message .= "2. Secure your account\n";
            $message .= "3. Request key revocation if needed";
            
            sendMessage($chatId, $message);
            return false;
        }
        
        // Get user ID
        $stmt = $db->prepare("SELECT id FROM users WHERE telegram_id = ?");
        $stmt->execute([$telegramId]);
        $user = $stmt->fetch();
        
        if (!$user) {
            sendMessage($chatId,
                "❌ <b>There is an error</b>\n\n"
               . "Failed to verify user data.\n"
               . "Please try again or contact Admin.");
            return false;
        }
        
        // Activate key
        $stmt = $db->prepare("UPDATE access_keys SET is_used = 1, user_id = ?, used_at = NOW() WHERE id = ?");
        $stmt->execute([$user['id'], $keyData['id']]);
        
        $message = "🎉 <b>WELCOME TO PREMIUM!</b> 🎉\n";
        $message .= str_repeat("═", 30) . "\n";
        $message .= "✨ Your premium membership has been successfully activated!\n\n";
        $message .= "🆔 <b>Member ID:</b> PREM-" . $keyData['id'] . "\n";
        $message .= "📅 <b>Activated:</b> " . date('d M Y, H:i') . "\n";
        $message .= "🔑 <b>Key:</b> <code>$formattedKey</code>\n\n";
        $message .= "🌟 <b>Your Benefits:</b>\n";
        $message .= "• Exclusive content access\n";
        $message .= "• Priority downloads\n";
        $message .= "• VIP support channel\n";
        $message .= "• Special member privileges\n\n";
        $message .= "👉 <i>Use /premium to access your member area</i>";

        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🚀 ENTER PREMIUM AREA', 'callback_data' => 'premium'],
                    ['text' => '📚 MEMBER GUIDE', 'callback_data' => 'premium_guide']
                ],
                [
                    ['text' => '🆘 SUPPORT', 'url' => 'https://t.me/rinda_escobar']
               ]
            ]
        ];

        sendMessage($chatId, $message, $keyboard);

        logBotActivity('key_activation', "User ID: $telegramId activated key: $key");
        return true;
    } catch (PDOException $e) {
        logBotActivity('key_activation_error', $e->getMessage());
        $message = "⚠️ <b>SYSTEM ERROR</b> ⚠️\n";
        $message .= str_repeat("═", 30) . "\n";
        $message .= "We encountered an issue processing your key\n\n";
        $message .= "🔧 <b>Technical Details:</b>\n";
        $message .= "• Error: Database connection issue\n";
        $message .= "• Time: " . date('H:i:s') . "\n\n";
        $message .= "🔄 Please try again in a few minutes\n";
        $message .= "📞 Contact @rinda_escobar if problem persists";
        
        sendMessage($chatId, $message);
        return false;
    }
}

/**
 * Tampilkan daftar file
 */
function showFiles($chatId, $userId, $db, $isPremium = false, $categoryId = null) {
    try {
        if ($isPremium && !hasAccessToMemberArea($userId, $db)) {
            showMemberAreaInfo($chatId);
            return;
        }
        
        $query = "SELECT f.*, c.name as category_name 
                  FROM files f
                  LEFT JOIN categories c ON f.category_id = c.id
                  WHERE f.is_premium = ?";
        
        $params = [$isPremium ? 1 : 0];
        
        if ($categoryId) {
            $query .= " AND f.category_id = ?";
            $params[] = $categoryId;
        }
        
        $query .= " ORDER BY f.created_at DESC LIMIT 20";
        
        $stmt = $db->prepare($query);
        $stmt->execute($params);
        $files = $stmt->fetchAll();
        
        if (empty($files)) {
            $message = $isPremium ? "📭 *Premium Folder is Empty*\n\nSorry, there are no premium files available at this time." 
              : "📂 *Empty Folder*\n\nNo files can be accessed yet.";
            if ($categoryId) {
                $message = "🗂 *Empty Category*\n\nThere are no files in this category yet.";
            }
            sendMessage($chatId, $message, getBackKeyboard($isPremium ? 'premium' : 'files'));
            return;
        }
        
        $message = $isPremium ? "📁 <b>PREMIUM FILES:</b>\n\n" : "📁 <b>PUBLIC FILES:</b>\n\n";
        if ($categoryId) {
            $message = "📁 <b>Category Files {$files[0]['category_name']}:</b>\n\n";
        }
        
        foreach ($files as $file) {
    // Format data
    $fileSize = formatFileSize($file['file_size']);
    $fileType = strtoupper($file['file_type']);
    $fileName = htmlspecialchars($file['file_name']);
    $fileDesc = !empty($file['description']) ? htmlspecialchars($file['description']) : null;
    $fileCategory = !empty($file['category_name']) ? '#' . htmlspecialchars($file['category_name']) : '<i>Tidak ada</i>';
    $downloadCmd = "/download_" . $file['id'];

    // Bangun tampilan
    $message .= "═══════════════════════════\n";
    $message .= "🖥️ <b>$fileName</b>\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    
    if ($fileDesc) {
        $message .= "📋 $fileDesc\n";
        $message .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    }
    
    $message .= "📌 <b>File Details:</b>\n";
    $message .= "├ 🏷️ Type: $fileType\n";
    $message .= "├ 📦 Size: $fileSize\n";
    $message .= "└ 📁 Category: $fileCategory\n";
    $message .= "━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
    $message .= "⬇️ Downloads: <code>$downloadCmd</code>\n\n";
}
        
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '🏠 Main Menu', 'callback_data' => 'help'],
                    ['text' => '📂 All Categories', 'callback_data' => 'categories:'.($isPremium ? '1' : '0')]
                ]
            ]
        ];
        
        sendMessage($chatId, $message, $keyboard);
    } catch (PDOException $e) {
        logBotActivity('show_files_error', $e->getMessage());
        sendMessage($chatId, "❌ An error occurred while retrieving file list.", getBackKeyboard($isPremium ? 'premium' : 'files'));
    }
}

/**
 * Tampilkan informasi kategori
 */
function showCategories($chatId, $userId, $db, $isPremium = false) {
    try {
        $query = "SELECT c.id, c.name, COUNT(f.id) as file_count 
                  FROM categories c
                  LEFT JOIN files f ON c.id = f.category_id AND f.is_premium = ?
                  GROUP BY c.id
                  ORDER BY c.name";
        
        $stmt = $db->prepare($query);
        $stmt->execute([$isPremium ? 1 : 0]);
        $categories = $stmt->fetchAll();

        if (empty($categories)) {
            sendMessage($chatId, "📂 *No Categories Yet*\n\nSorry, there are no file categories available at this time.", getBackKeyboard($isPremium ? 'premium' : 'files'));
            return;
        }

        $keyboard = ['inline_keyboard' => []];
        
        // Tambahkan tombol kategori
        foreach ($categories as $cat) {
            $keyboard['inline_keyboard'][] = [
                ['text' => "{$cat['name']} ({$cat['file_count']})", 
                 'callback_data' => "category:{$cat['id']}:".($isPremium ? '1' : '0')]
            ];
        }
        
        // Tambahkan tombol kembali
        $keyboard['inline_keyboard'][] = [
            ['text' => '◀️ Back', 'callback_data' => $isPremium ? 'premium' : 'files']
        ];

        sendMessage($chatId, "📁 <b>Select Category:</b>", $keyboard);
    } catch (PDOException $e) {
        logBotActivity('show_categories_error', $e->getMessage());
        sendMessage($chatId, "❌ An error occurred while retrieving category list.", getBackKeyboard($isPremium ? 'premium' : 'files'));
    }
}

/**
 * Tampilkan search 
 */
function handleSearch($chatId, $userId, $searchQuery, $db, $isPremium = false) {
    try {
        $query = "SELECT * FROM files 
                  WHERE (file_name LIKE ? OR description LIKE ?)
                  AND is_premium = ?
                  ORDER BY created_at DESC 
                  LIMIT 10";
        
        $stmt = $db->prepare($query);
        $stmt->execute(["%$searchQuery%", "%$searchQuery%", $isPremium ? 1 : 0]);
        $files = $stmt->fetchAll();

        if (empty($files)) {
            $message = "🔍 <b>Search Not Found</b>\n\n"
             . "No files match the keyword:\n"
             . "`" . $searchQuery . "`";
            if ($isPremium) {
                $message .= "\n\n🔎 <b>Premium Tips</b>\n"
                  . "Try searching in public files with:\n"
                  . "`/search " . $searchQuery . "`";
            }
            sendMessage($chatId, $message, getBackKeyboard($isPremium ? 'premium' : 'files'));
            return;
        }

        $message = "🔍 <b>Search Results for '$searchQuery':</b>\n\n";
        foreach ($files as $file) {
    // Format data - TIDAK DIUBAH
    $fileSize = formatFileSize($file['file_size']);
    $fileType = strtoupper($file['file_type']);
    $fileName = htmlspecialchars($file['file_name']);
    $fileDesc = !empty($file['description']) ? htmlspecialchars($file['description']) : null;
    $fileCategory = !empty($file['category_name']) ? '#' . htmlspecialchars($file['category_name']) : '<i>Tidak ada</i>';
    $downloadCmd = "/download_" . $file['id']; // COMMAND TETAP SAMA

    // Bangun tampilan - HANYA PERBAIKAN VISUAL
    $message .= "═══════════════════════════\n";
    $message .= "📂 <b>$fileName</b>\n";
    $message .= "───────────────────────────\n";
    
    if ($fileDesc) {
        $message .= "📋 $fileDesc\n";
        $message .= "───────────────────────────\n";
    }
    
    $message .= "🔍 <b>Detail File:</b>\n";
    $message .= "├ 🏷️ Type: $fileType\n";
    $message .= "├ 📦 Size: $fileSize\n";
    $message .= "└ 📁 Category: $fileCategory\n";
    $message .= "───────────────────────────\n";
    $message .= "⬇️ <code>$downloadCmd</code>\n\n";
}

        sendMessage($chatId, $message, getBackKeyboard($isPremium ? 'premium' : 'files'));
    } catch (PDOException $e) {
        logBotActivity('search_error', $e->getMessage());
        sendMessage($chatId, "❌ An error occurred while searching.", getBackKeyboard($isPremium ? 'premium' : 'files'));
    }
}

/**
 * Tampilkan informasi member area
 */
function showMemberAreaInfo($chatId) {
    $message = "🌟 <b>PREMIUM MEMBER PORTAL</b> 🌟\n";
    $message .= str_repeat("═", 27) . "\n";
    $message .= "🔒 This area requires premium membership\n\n";
    $message .= "💎 <b>Membership Benefits:</b>\n";
    $message .= "• Exclusive content files\n";
    $message .= "• Early access to new releases\n";
    $message .= "• Priority customer support\n";
    $message .= "• VIP member community\n\n";
    $message .= "🔑 <b>How to Join:</b>\n";
    $message .= "1. Obtain an activation key\n";
    $message .= "2. Use <code>/activate YOUR_KEY</code>\n";
    $message .= "3. Enjoy premium access!\n\n";
    $message .= "📌 <i>Limited slots available</i>";

    $keyboard = [
        'inline_keyboard' => [
            [
                ['text' => '🔑 ACTIVATE KEY', 'callback_data' => 'activate_key'],
                ['text' => '💳 GET MEMBERSHIP', 'url' => 'https://t.me/rinda_escobar']
            ],
            [
                ['text' => '❓ FAQ', 'callback_data' => 'premium_faq']
            ]
        ]
    ];

    sendMessage($chatId, $message, $keyboard);
}

/**
 * Proses download file
 */
function processFileDownload($chatId, $telegramId, $fileId, $db)
{
    try {
        // Get file info
        $stmt = $db->prepare("SELECT * FROM files WHERE id = ?");
        $stmt->execute([$fileId]);
        $file = $stmt->fetch();
        
        if (!$file) {
            sendMessage($chatId, "📭 File Not Found");
            return;
        }
        
        // Check if premium file
        if ($file['is_premium'] && !hasAccessToMemberArea($telegramId, $db)) {
            showMemberAreaInfo($chatId);
            return;
        }
        
        // Cek rate limit - batas 10 download per hari untuk user biasa, 30 untuk premium
        $maxDownloads = hasAccessToMemberArea($telegramId, $db) ? 50 : 20;
        $limitCheck = checkDownloadLimit($telegramId, $maxDownloads, $db);
        
        if ($limitCheck && $limitCheck['limit_reached']) {
            $resetTime = date('H:i', $limitCheck['next_reset']);
            $message = "⏳ <b>DAILY DOWNLOAD LIMIT</b>\n";
            $message .= "═══════════════════════════\n";
            $message .= "You have reached limit:\n";
            $message .= "• {$limitCheck['max_allowed']} downloads/day\n";
            $message .= "• Reset <b>$resetTime WIB</b>\n\n";
            /*
            if (!hasAccessToMemberArea($telegramId, $db)) {
                $message .= "💎 <i>Upgrade to premium for:</i>\n";
                $message .= "▸ Higher download limit\n";
                $message .= "▸ Access exclusive files\n";
                $message .= "▸ Download priority\n\n";
                $message .= "🔓 Type /premium for info";
            }*/
            
            sendMessage($chatId, $message);
            return;
        }
        
        $filePath = __DIR__ . '/../' . $file['file_path'];
        
        if (!file_exists($filePath)) {
            sendMessage($chatId, "📭 File Not Available");
            return;
        }
        
        logDownload($telegramId, $fileId, $db);
        
        // Update download count
        $stmt = $db->prepare("UPDATE files SET download_count = download_count + 1 WHERE id = ?");
        $stmt->execute([$fileId]);
        
        // Send file
        sendDocument($chatId, $filePath, "📥 <b>{$file['file_name']}</b>");
        
        // Tampilkan info batas download setelah download berhasil
        $maxDownloads = hasAccessToMemberArea($telegramId, $db) ? 50 : 20;
        $today = date('Y-m-d 00:00:00');
        
        // Dapatkan user ID dari telegram ID
        $stmt = $db->prepare("SELECT id FROM users WHERE telegram_id = ?");
        $stmt->execute([$telegramId]);
        $user = $stmt->fetch();
        $userId = $user['id'];
        
        // Hitung jumlah download hari ini setelah download ini
        $stmt = $db->prepare("
            SELECT COUNT(*) as download_count 
            FROM download_logs 
            WHERE user_id = ? AND created_at >= ?
        ");
        $stmt->execute([$userId, $today]);
        $result = $stmt->fetch();
        $downloadCount = (int)$result['download_count'];
        
        $infoMessage = "📊 <b>Download Count</b>\n"
             . "━━━━━━━━━━━━━━━━━━\n"
             . "• Today: <b>{$downloadCount}/{$maxDownloads}</b>\n"
             . "• Reset: <code>" . date('H:i') . " WIB</code>";
        sendMessage($chatId, $infoMessage);
        
        logBotActivity('file_download', "User: $telegramId, File: $fileId");
    } catch (PDOException $e) {
        logBotActivity('download_error', $e->getMessage());
        sendMessage($chatId, "An error occurred while downloading file.");
    }
}

/**
 * Handle file upload
 */
function handleFileUpload($message, $db)
{
    $chatId = $message['chat']['id'];
    $userId = $message['from']['id'];
    $document = $message['document'];
    $fileName = $document['file_name'];
    $fileId = $document['file_id'];
    $fileSize = $document['file_size'];
    $fileType = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
    
    // Only allow admin to upload
    if ($userId != $_ENV['ADMIN_TELEGRAM_ID']) {
        sendMessage($chatId, "⛔ <b>Access denied</b>\nOnly admins can upload files.");
        return;
    }
    
    // Download file from Telegram
    $url = "https://api.telegram.org/bot" . $_ENV['BOT_TOKEN'] . "/getFile";
    $fileInfo = makeRequest($url, ['file_id' => $fileId]);
    
    if (!isset($fileInfo['result']['file_path'])) {
        sendMessage($chatId, "❌ Failed to Load File Information");
        return;
    }
    
    $filePath = $fileInfo['result']['file_path'];
    $downloadUrl = "https://api.telegram.org/file/bot" . $_ENV['BOT_TOKEN'] . "/" . $filePath;
    
    // Create directory if not exists
    $uploadDir = __DIR__ . '/../public/files/' . date('Y-m-d');
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Generate unique filename
    $uniqueFilename = uniqid() . '_' . $fileName;
    $savePath = $uploadDir . '/' . $uniqueFilename;
    
    // Download file
    $fileContent = file_get_contents($downloadUrl);
    if ($fileContent === false) {
        sendMessage($chatId, "❌ <b>Download Error</b>\nFailed to retrieve file");
        return;
    }
    
    if (file_put_contents($savePath, $fileContent) === false) {
        sendMessage($chatId, "❌ <b>Save Error</b>\nFailed to save file");
        return;
    }
    
    // Save to database
    try {
        $relativeFilePath = 'public/files/' . date('Y-m-d') . '/' . $uniqueFilename;
        $stmt = $db->prepare("
            INSERT INTO files (file_name, file_path, file_size, file_type, is_premium) 
            VALUES (?, ?, ?, ?, 0)
        ");
        $stmt->execute([$fileName, $relativeFilePath, $fileSize, $fileType]);
        
        sendMessage($chatId, "✅ File berhasil diupload!\n\nNama: $fileName\nUkuran: " . formatFileSize($fileSize));
        
        // Ask if file is premium
        $keyboard = [
            'inline_keyboard' => [
                [
                    ['text' => '✅ Ya, Jadikan Premium', 'callback_data' => 'set_premium:' . $db->lastInsertId()],
                    ['text' => '❌ Tidak', 'callback_data' => 'help']
                ]
            ]
        ];
        
        sendMessage($chatId, "Apakah file ini untuk member premium?", $keyboard);
    } catch (PDOException $e) {
        logBotActivity('file_upload_error', $e->getMessage());
        sendMessage($chatId, "Terjadi kesalahan saat menyimpan data file.");
    }
}

/**
 * Format ukuran file
 */
function formatFileSize($bytes)
{
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}

/**
 * Catat aktivitas bot
 */
function logBotActivity($action, $details = null)
{
    try {
        // Simpan di file log
        $logFile = __DIR__ . '/../logs/bot_' . date('Y-m-d') . '.log';
        $logDir = dirname($logFile);
        
        if (!is_dir($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        $logData = date('Y-m-d H:i:s') . " | $action | $details\n";
        file_put_contents($logFile, $logData, FILE_APPEND);
        
        // Jika perlu, simpan juga di database
        // Implementasi logging ke database bisa ditambahkan di sini
    } catch (Exception $e) {
        // Ignore logging errors
    }
}

/**
 * Tampilkan bantuan
 */
function showHelp($chatId) {
    $message = "✨ <b>BOT USER GUIDE</b> ✨\n";
    $message .= "═══════════════════════════\n";
    $message .= "RINDA'S Bot.\n\n";
    
    $message .= "📋 <b>COMMAND LIST</b>\n";
    $message .= "▸ /start - Starting bot\n";
    $message .= "▸ /files - Browse public files\n";
    //$message .= "▸ /premium - Access exclusive content\n";
    $message .= "▸ /categories - Browse by category\n";
    $message .= "▸ /search [query] - Find specific files\n";
    $message .= "▸ /help - View this guide\n\n";
    
    $message .= "💡 <i>Use menu button for navigation</i>\n";
    $message .= "🛠 Need help? Contact @rinda_escobar";

    sendMessage($chatId, $message, getMainKeyboard());
}

/**
 * Broadcast pesan ke pengguna
 */
function broadcastMessage($message, $db, $onlyPremium = false) {
    try {
        // Pastikan $db adalah objek PDO yang valid
        if (!($db instanceof PDO)) {
            throw new Exception("Koneksi database tidak valid");
        }
        
        // Ambil daftar pengguna yang akan menerima broadcast
        if ($onlyPremium) {
            $stmt = $db->query("
                SELECT DISTINCT u.telegram_id 
                FROM users u
                JOIN access_keys ak ON u.id = ak.user_id
                WHERE u.is_blocked = 0 AND ak.is_used = 1
            ");
        } else {
            $stmt = $db->query("
                SELECT telegram_id FROM users 
                WHERE is_blocked = 0
            ");
        }
        
        $users = $stmt->fetchAll();
        $successCount = 0;
        $failCount = 0;
        
        foreach ($users as $user) {
            $result = sendMessage($user['telegram_id'], $message);
            if ($result && isset($result['ok']) && $result['ok']) {
                $successCount++;
                // Tunggu sebentar untuk menghindari rate limiting
                usleep(100000); // 100ms
            } else {
                $failCount++;
            }
        }
        
        logBotActivity('broadcast_message', "Sent to: $successCount users, Failed: $failCount, Premium only: " . ($onlyPremium ? 'Yes' : 'No'));
        
        return [
            'total' => count($users),
            'success' => $successCount,
            'failed' => $failCount
        ];
    } catch (PDOException $e) {
        logBotActivity('broadcast_error', $e->getMessage());
        return false;
    } catch (Exception $e) {
        logBotActivity('broadcast_error', $e->getMessage());
        return false;
    }
}

/**
 * Handle image to base64 conversion
 */
function handleImageToBase64($message, $db) {
    $chatId = $message['chat']['id'];
    $userId = $message['from']['id'];
    
    // Check if message contains photo
    if (!isset($message['photo'])) {
        sendMessage($chatId, "📸 Please send an image to convert to Base64");
        return;
    }
    
    // Get the highest quality photo
    $photo = end($message['photo']);
    $fileId = $photo['file_id'];
    
    // Download file from Telegram
    $url = "https://api.telegram.org/bot" . $_ENV['BOT_TOKEN'] . "/getFile";
    $fileInfo = makeRequest($url, ['file_id' => $fileId]);
    
    if (!isset($fileInfo['result']['file_path'])) {
        sendMessage($chatId, "❌ Failed to process image");
        return;
    }
    
    $filePath = $fileInfo['result']['file_path'];
    $downloadUrl = "https://api.telegram.org/file/bot" . $_ENV['BOT_TOKEN'] . "/" . $filePath;
    
    // Create directory if not exists
    $uploadDir = __DIR__ . '/../public/tools/' . date('Y-m-d');
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
    }
    
    // Generate unique filename
    $uniqueFilename = 'base64_' . uniqid() . '.txt';
    $savePath = $uploadDir . '/' . $uniqueFilename;
    
    // Download image
    $imageContent = file_get_contents($downloadUrl);
    if ($imageContent === false) {
        sendMessage($chatId, "❌ Failed to download image");
        return;
    }
    
    // Convert to base64
    $base64Content = base64_encode($imageContent);
    
    // Save to file
    if (file_put_contents($savePath, $base64Content) === false) {
        sendMessage($chatId, "❌ Failed to save conversion");
        return;
    }
    
    // Get user ID from database
    $stmt = $db->prepare("SELECT id FROM users WHERE telegram_id = ?");
    $stmt->execute([$userId]);
    $user = $stmt->fetch();
    
    if ($user) {
        // Log conversion
        $relativePath = 'public/tools/' . date('Y-m-d') . '/' . $uniqueFilename;
        $stmt = $db->prepare("INSERT INTO tool_conversions (user_id, tool_type, file_path) VALUES (?, ?, ?)");
        $stmt->execute([$user['id'], 'image_to_base64', $relativePath]);
    }
    
    // Send file to user
    sendDocument($chatId, $savePath, "🔤 Base64 encoded image");
    
    // Delete file after sending
    unlink($savePath);
}

/**
 * Memeriksa batas download pengguna
 * @param int $telegramId ID Telegram pengguna
 * @param int $maxDownloadsPerDay Jumlah maksimal download per hari
 * @param PDO $db Koneksi database
 * @return bool|array false jika tidak melebihi batas, array info jika melebihi
 */
function checkDownloadLimit($telegramId, $maxDownloadsPerDay, $db) 
{
    try {
        // Dapatkan user ID dari telegram ID
        $stmt = $db->prepare("SELECT id FROM users WHERE telegram_id = ?");
        $stmt->execute([$telegramId]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return false; // Pengguna tidak ditemukan
        }
        
        $userId = $user['id'];
        
        // Hitung jumlah download hari ini
        $today = date('Y-m-d 00:00:00');
        $stmt = $db->prepare("
            SELECT COUNT(*) as download_count 
            FROM download_logs 
            WHERE user_id = ? AND created_at >= ?
        ");
        $stmt->execute([$userId, $today]);
        $result = $stmt->fetch();
        
        $downloadCount = (int)$result['download_count'];
        
        // Periksa apakah melebihi batas
        if ($downloadCount >= $maxDownloadsPerDay) {
            return [
                'limit_reached' => true,
                'downloads_today' => $downloadCount,
                'max_allowed' => $maxDownloadsPerDay,
                'next_reset' => strtotime('tomorrow midnight')
            ];
        }
        
        return false; // Tidak melebihi batas
    } catch (PDOException $e) {
        logBotActivity('check_limit_error', $e->getMessage());
        return false; // Asumsikan tidak ada batasan jika terjadi error
    }
}

/**
 * Mencatat download ke dalam log
 */
function logDownload($telegramId, $fileId, $db) 
{
    try {
        // Dapatkan user ID dari telegram ID
        $stmt = $db->prepare("SELECT id FROM users WHERE telegram_id = ?");
        $stmt->execute([$telegramId]);
        $user = $stmt->fetch();
        
        if (!$user) {
            return false;
        }
        
        $userId = $user['id'];
        
        // Catat download
        $stmt = $db->prepare("
            INSERT INTO download_logs (user_id, file_id) 
            VALUES (?, ?)
        ");
        $stmt->execute([$userId, $fileId]);
        
        return true;
    } catch (PDOException $e) {
        logBotActivity('log_download_error', $e->getMessage());
        return false;
    }
}


function sendChatAction($chatId, $action) {
    $actions = ['typing', 'upload_photo', 'record_video', 'upload_video', 
                'record_audio', 'upload_audio', 'upload_document', 'find_location'];
    
    if (!in_array($action, $actions)) {
        $action = 'typing';
    }
    
    $url = "https://api.telegram.org/bot" . $_ENV['BOT_TOKEN'] . "/sendChatAction";
    $params = [
        'chat_id' => $chatId,
        'action' => $action
    ];
    
    return makeRequest($url, $params);
}

function deleteMessage($chatId, $messageId) {
    $url = "https://api.telegram.org/bot" . $_ENV['BOT_TOKEN'] . "/deleteMessage";
    $params = [
        'chat_id' => $chatId,
        'message_id' => $messageId
    ];
    
    return makeRequest($url, $params);
}

function editMessageText($chatId, $messageId, $text, $replyMarkup = null) {
    $url = "https://api.telegram.org/bot" . $_ENV['BOT_TOKEN'] . "/editMessageText";
    $params = [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    
    if ($replyMarkup !== null) {
        $params['reply_markup'] = json_encode($replyMarkup);
    }
    
    return makeRequest($url, $params);
}