<?php
/**
 * Admin Panel - Generate Key
 * File ini untuk manajemen dan generate key untuk akses member area
 */

// Include auth check
require_once 'auth_check.php';

// Process actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        $action = $_POST['action'];
        
        try {
            switch ($action) {
                case 'generate':
                    $quantity = isset($_POST['quantity']) ? (int)$_POST['quantity'] : 1;
                    $quantity = min(max($quantity, 1), 100); // Min 1, Max 100
                    
                    $keys = [];
                    for ($i = 0; $i < $quantity; $i++) {
                        $keyValue = generateUniqueKey($db);
                        $stmt = $db->prepare("INSERT INTO access_keys (key_value) VALUES (?)");
                        $stmt->execute([$keyValue]);
                        $keys[] = $keyValue;
                    }
                    
                    $successMessage = "Berhasil generate $quantity key baru";
                    break;
                    
                case 'delete':
                    $keyId = isset($_POST['key_id']) ? (int)$_POST['key_id'] : 0;
                    
                    // Check if key is used
                    $stmt = $db->prepare("SELECT is_used FROM access_keys WHERE id = ?");
                    $stmt->execute([$keyId]);
                    $key = $stmt->fetch();
                    
                    if ($key && $key['is_used']) {
                        $errorMessage = "Tidak dapat menghapus key yang sudah digunakan";
                    } else {
                        $stmt = $db->prepare("DELETE FROM access_keys WHERE id = ? AND is_used = 0");
                        $stmt->execute([$keyId]);
                        $successMessage = "Key berhasil dihapus";
                    }
                    break;
                    
                case 'delete_selected':
                    if (isset($_POST['selected_keys']) && is_array($_POST['selected_keys'])) {
                        $selectedKeys = $_POST['selected_keys'];
                        
                        // Delete only unused keys
                        $placeholders = implode(',', array_fill(0, count($selectedKeys), '?'));
                        $stmt = $db->prepare("DELETE FROM access_keys WHERE id IN ($placeholders) AND is_used = 0");
                        $stmt->execute($selectedKeys);
                        
                        $deletedCount = $stmt->rowCount();
                        $successMessage = "Berhasil menghapus $deletedCount key";
                    } else {
                        $errorMessage = "Tidak ada key yang dipilih";
                    }
                    break;
            }
        } catch (PDOException $e) {
            $errorMessage = "Database error: " . $e->getMessage();
        }
    }
}

// Filter settings
$filter = isset($_GET['filter']) ? $_GET['filter'] : 'all';
$search = isset($_GET['search']) ? $_GET['search'] : '';

// Build query based on filter
$query = "SELECT ak.*, u.username, u.first_name, u.last_name, u.telegram_id FROM access_keys ak LEFT JOIN users u ON ak.user_id = u.id WHERE 1=1";

if ($filter === 'used') {
    $query .= " AND ak.is_used = 1";
} elseif ($filter === 'unused') {
    $query .= " AND ak.is_used = 0";
}

if (!empty($search)) {
    $query .= " AND (ak.key_value LIKE :search OR u.username LIKE :search OR u.first_name LIKE :search OR u.last_name LIKE :search)";
}

$query .= " ORDER BY ak.created_at DESC";

// Pagination
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$perPage = 20;
$offset = ($page - 1) * $perPage;

// Get total count for pagination
$countQuery = str_replace("SELECT ak.*, u.username, u.first_name, u.last_name, u.telegram_id", "SELECT COUNT(*)", $query);
$stmt = $db->prepare($countQuery);

if (!empty($search)) {
    $stmt->bindValue(':search', "%$search%", PDO::PARAM_STR);
}

$stmt->execute();
$totalKeys = $stmt->fetchColumn();
$totalPages = ceil($totalKeys / $perPage);

// Get keys with limit
$query .= " LIMIT :offset, :perPage";
$stmt = $db->prepare($query);

if (!empty($search)) {
    $stmt->bindValue(':search', "%$search%", PDO::PARAM_STR);
}

$stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
$stmt->bindValue(':perPage', $perPage, PDO::PARAM_INT);
$stmt->execute();
$keys = $stmt->fetchAll();

// Get key statistics
$stmt = $db->query("SELECT COUNT(*) as total FROM access_keys");
$totalAllKeys = $stmt->fetch()['total'];

$stmt = $db->query("SELECT COUNT(*) as total FROM access_keys WHERE is_used = 1");
$totalUsedKeys = $stmt->fetch()['total'];

$stmt = $db->query("SELECT COUNT(*) as total FROM access_keys WHERE is_used = 0");
$totalUnusedKeys = $stmt->fetch()['total'];

// Page title
$pageTitle = "Generate Key";

/**
 * Generate unique random key
 */
function generateUniqueKey($db, $length = 16) {
    do {
        $key = bin2hex(random_bytes($length / 2)); // Generate random key
        
        // Check if key exists
        $stmt = $db->prepare("SELECT COUNT(*) FROM access_keys WHERE key_value = ?");
        $stmt->execute([$key]);
        $exists = $stmt->fetchColumn() > 0;
    } while ($exists);
    
    return $key;
}
?>

<?php include 'include/header.php'; ?>

<div class="container-fluid px-4">
    <h1 class="mt-4">Generate Key Member</h1>
    <ol class="breadcrumb mb-4">
        <li class="breadcrumb-item"><a href="dashboard.php">Dashboard</a></li>
        <li class="breadcrumb-item active">Generate Key</li>
    </ol>
    
    <?php if (isset($successMessage)): ?>
        <div class="alert alert-success" role="alert">
            <?php echo $successMessage; ?>
        </div>
    <?php endif; ?>
    
    <?php if (isset($errorMessage)): ?>
        <div class="alert alert-danger" role="alert">
            <?php echo $errorMessage; ?>
        </div>
    <?php endif; ?>
    
    <!-- Statistics cards -->
    <div class="row mb-4">
        <div class="col-xl-4 col-md-6">
            <div class="card bg-primary text-white mb-4">
                <div class="card-body">
                    <h4 class="mb-0"><?php echo $totalAllKeys; ?></h4>
                    <div>Total Key</div>
                </div>
            </div>
        </div>
        <div class="col-xl-4 col-md-6">
            <div class="card bg-success text-white mb-4">
                <div class="card-body">
                    <h4 class="mb-0"><?php echo $totalUnusedKeys; ?></h4>
                    <div>Key Belum Digunakan</div>
                </div>
            </div>
        </div>
        <div class="col-xl-4 col-md-6">
            <div class="card bg-warning text-white mb-4">
                <div class="card-body">
                    <h4 class="mb-0"><?php echo $totalUsedKeys; ?></h4>
                    <div>Key Sudah Digunakan</div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Generate key form -->
    <div class="card mb-4">
        <div class="card-header">
            <i class="bi bi-key me-1"></i>
            Generate Key Baru
        </div>
        <div class="card-body">
            <form action="" method="POST" class="row g-3">
                <div class="col-md-6">
                    <label for="keyQuantity" class="form-label">Jumlah Key</label>
                    <input type="number" class="form-control" id="keyQuantity" name="quantity" min="1" max="100" value="1">
                    <div class="form-text">Masukkan jumlah key yang ingin di-generate (maksimal 100)</div>
                </div>
                <div class="col-12">
                    <button type="submit" name="action" value="generate" class="btn btn-primary">
                        <i class="bi bi-key"></i> Generate Key
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Keys list -->
    <div class="card mb-4">
        <div class="card-header">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <i class="bi bi-list me-1"></i>
                    Daftar Key
                </div>
                <div>
                    <a href="?filter=all" class="btn btn-sm <?php echo $filter === 'all' ? 'btn-primary' : 'btn-outline-primary'; ?>">Semua</a>
                    <a href="?filter=unused" class="btn btn-sm <?php echo $filter === 'unused' ? 'btn-success' : 'btn-outline-success'; ?>">Belum Digunakan</a>
                    <a href="?filter=used" class="btn btn-sm <?php echo $filter === 'used' ? 'btn-warning' : 'btn-outline-warning'; ?>">Sudah Digunakan</a>
                </div>
            </div>
        </div>
        <div class="card-body">
            <!-- Search form -->
            <div class="row mb-3">
                <div class="col-md-6">
                    <form action="" method="GET" class="d-flex">
                        <input type="hidden" name="filter" value="<?php echo $filter; ?>">
                        <input type="text" name="search" id="searchInput" class="form-control me-2" placeholder="Cari key..." value="<?php echo htmlspecialchars($search); ?>">
                        <button type="submit" class="btn btn-primary">Cari</button>
                    </form>
                </div>
                <div class="col-md-6 text-end">
                    <button id="copySelectedBtn" class="btn btn-secondary me-2" disabled>
                        <i class="bi bi-clipboard"></i> Copy Selected
                    </button>
                    <button id="deleteSelectedBtn" class="btn btn-danger" disabled>
                        <i class="bi bi-trash"></i> Delete Selected
                    </button>
                </div>
            </div>
            
            <!-- Keys table -->
            <form id="keysForm" action="" method="POST">
                <input type="hidden" name="action" value="delete_selected">
                <div class="table-responsive">
                    <table class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="selectAll">
                                    </div>
                                </th>
                                <th>Key</th>
                                <th>Status</th>
                                <th>Digunakan Oleh</th>
                                <th>Dibuat</th>
                                <th>Digunakan Pada</th>
                                <th>Aksi</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($keys)): ?>
                                <tr>
                                    <td colspan="7" class="text-center">Tidak ada data key</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($keys as $key): ?>
                                    <tr>
                                        <td>
                                            <div class="form-check">
                                                <input class="form-check-input key-checkbox" type="checkbox" 
                                                       name="selected_keys[]" value="<?php echo $key['id']; ?>"
                                                       <?php echo $key['is_used'] ? 'disabled' : ''; ?>>
                                            </div>
                                        </td>
                                        <td>
                                            <code class="key-value"><?php echo $key['key_value']; ?></code>
                                            <button type="button" class="btn btn-sm btn-link copy-btn" 
                                                   data-key="<?php echo $key['key_value']; ?>">
                                                <i class="bi bi-clipboard"></i>
                                            </button>
                                        </td>
                                        <td>
                                            <?php if ($key['is_used']): ?>
                                                <span class="badge bg-warning">Sudah Digunakan</span>
                                            <?php else: ?>
                                                <span class="badge bg-success">Belum Digunakan</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($key['is_used']): ?>
                                                <?php
                                                    if (!empty($key['username'])) {
                                                        echo '@' . htmlspecialchars($key['username']);
                                                    } elseif (!empty($key['first_name'])) {
                                                        echo htmlspecialchars($key['first_name'] . ' ' . $key['last_name']);
                                                    } elseif (!empty($key['telegram_id'])) {
                                                        echo 'ID: ' . $key['telegram_id'];
                                                    } else {
                                                        echo '<span class="text-muted">-</span>';
                                                    }
                                                ?>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo date('d/m/Y H:i', strtotime($key['created_at'])); ?></td>
                                        <td>
                                            <?php if ($key['used_at']): ?>
                                                <?php echo date('d/m/Y H:i', strtotime($key['used_at'])); ?>
                                            <?php else: ?>
                                                <span class="text-muted">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (!$key['is_used']): ?>
                                                <form action="" method="POST" class="d-inline">
                                                    <input type="hidden" name="key_id" value="<?php echo $key['id']; ?>">
                                                    <button type="submit" name="action" value="delete" class="btn btn-sm btn-danger btn-delete">
                                                        <i class="bi bi-trash"></i>
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </form>
            
            <!-- Pagination -->
            <?php if ($totalPages > 1): ?>
                <nav aria-label="Page navigation">
                    <ul class="pagination justify-content-center">
                        <li class="page-item <?php echo ($page <= 1) ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?filter=<?php echo $filter; ?>&page=<?php echo $page-1; ?>&search=<?php echo urlencode($search); ?>">Previous</a>
                        </li>
                        
                        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                            <li class="page-item <?php echo ($page == $i) ? 'active' : ''; ?>">
                                <a class="page-link" href="?filter=<?php echo $filter; ?>&page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>"><?php echo $i; ?></a>
                            </li>
                        <?php endfor; ?>
                        
                        <li class="page-item <?php echo ($page >= $totalPages) ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?filter=<?php echo $filter; ?>&page=<?php echo $page+1; ?>&search=<?php echo urlencode($search); ?>">Next</a>
                        </li>
                    </ul>
                </nav>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- JavaScript for key actions -->
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Copy button functionality
    document.querySelectorAll('.copy-btn').forEach(function(button) {
        button.addEventListener('click', function() {
            const keyValue = this.getAttribute('data-key');
            navigator.clipboard.writeText(keyValue).then(function() {
                // Show tooltip or notification
                alert('Key berhasil disalin: ' + keyValue);
            });
        });
    });
    
    // Select all checkbox
    const selectAllCheckbox = document.getElementById('selectAll');
    const keyCheckboxes = document.querySelectorAll('.key-checkbox:not([disabled])');
    
    selectAllCheckbox.addEventListener('change', function() {
        keyCheckboxes.forEach(function(checkbox) {
            checkbox.checked = selectAllCheckbox.checked;
        });
        updateButtonState();
    });
    
    keyCheckboxes.forEach(function(checkbox) {
        checkbox.addEventListener('change', function() {
            updateButtonState();
            
            // Update selectAll state
            let allChecked = true;
            keyCheckboxes.forEach(function(cb) {
                if (!cb.checked) allChecked = false;
            });
            selectAllCheckbox.checked = allChecked;
        });
    });
    
    // Update button state based on selections
    function updateButtonState() {
        const hasSelection = Array.from(keyCheckboxes).some(checkbox => checkbox.checked);
        document.getElementById('copySelectedBtn').disabled = !hasSelection;
        document.getElementById('deleteSelectedBtn').disabled = !hasSelection;
    }
    
    // Copy selected keys
    document.getElementById('copySelectedBtn').addEventListener('click', function() {
        const selectedKeys = [];
        document.querySelectorAll('.key-checkbox:checked').forEach(function(checkbox) {
            const keyValue = checkbox.closest('tr').querySelector('.key-value').textContent;
            selectedKeys.push(keyValue);
        });
        
        if (selectedKeys.length > 0) {
            navigator.clipboard.writeText(selectedKeys.join('\n')).then(function() {
                alert('Keys berhasil disalin:\n' + selectedKeys.join('\n'));
            });
        }
    });
    
    // Delete selected keys confirmation
    document.getElementById('deleteSelectedBtn').addEventListener('click', function() {
        const selectedCount = document.querySelectorAll('.key-checkbox:checked').length;
        if (selectedCount > 0) {
            if (confirm(`Apakah Anda yakin ingin menghapus ${selectedCount} key?`)) {
                document.getElementById('keysForm').submit();
            }
        }
    });
    
    // Delete button confirmation
    document.querySelectorAll('.btn-delete').forEach(function(button) {
        button.addEventListener('click', function(e) {
            if (!confirm('Apakah Anda yakin ingin menghapus key ini?')) {
                e.preventDefault();
            }
        });
    });
});
</script>

<?php include 'include/footer.php'; ?>
